package container;

import geom.GCoordSystem;
import geom.GObject;
import geom.GView;

import java.util.ArrayList;
import java.util.TreeSet;
import java.util.Iterator;
import java.lang.reflect.*;
import java.util.Vector;
import java.util.logging.*;

/**
 * \ingroup Container 
 * \brief Kласс для хранения графических объектов
 * 
 * Основные операции \ref container.GeomContainer#add() "add", 
 * \ref container.GeomContainer#getSelectedObjects() "getSelectedObjects" и 
 * \ref container.GeomContainer#getObjectsTypeOf() "getObjectsTypeOf"
 * 
 * Пример обхода контейнера: 
 * \code 
 * Iterator<GObject> iterator = iterator();
 * while (iterator.hasNext()) 
 * { 
 * 		GObject obj = iterator.next(); 
 * 		//...действия с объектом obj 
 * } 
 * \endcode
 * 
 * Преобразование в массив - функция суперкласса - toArray()
 */
public class GeomContainer extends TreeSet<GObject>
{
	private ArrayList<IGeomContainerListener> listeners = new ArrayList<IGeomContainerListener>();

	private GCoordSystem gcs;

	public void addListener(IGeomContainerListener listener)
	{
		listeners.add(listener);
	}

	public void removeListener(IGeomContainerListener listener)
	{
		listeners.remove(listener);
	}

	private void fireListeners()
	{
		for (IGeomContainerListener listener : listeners)
		{
			listener.onDataChanged(this);
		}
	}

	/**
	 * \ingroup Container 
	 * \brief Возвращает новый контейнер, со всеми выделенными объектами
	 */
	public GeomContainer getSelectedObjects()
	{
		GeomContainer selected = new GeomContainer(gcs);
		Iterator<GObject> iterator = iterator();
		while (iterator.hasNext())
		{
			GObject obj = iterator.next();
			if (obj.view().isSelected())
			{
				selected.add(obj);
			}
		}
		return selected;
	}

	/**
	 * \ingroup Container 
	 * \brief Возвращает новый контейнер со всеми объектами данного класса, например Point.class.
	 */
	public GeomContainer getObjectsTypeOf(Class c)
	{
		GeomContainer objs = new GeomContainer(gcs);
		Iterator<GObject> iterator = iterator();
		while (iterator.hasNext())
		{
			GObject obj = iterator.next();
			if (obj.getClass().equals(c))
			{
				objs.add(obj);
			}
		}
		return objs;
	}

	public GeomContainer(GCoordSystem gcs) {
		this.gcs = gcs;
	}

	/**
	 * \ingroup Container
	 */
	public boolean add(GObject obj)
	{
		if (super.add(obj))
		{
			fireListeners();
			return true;
		}
		return false;
	}

	public void remove(GObject obj)
	{
		super.remove(obj);
		fireListeners();
	}

	public void draw(GCoordSystem gcs)
	{
		Iterator<GObject> iterator = iterator();
		while (iterator.hasNext())
		{
			iterator.next().draw(gcs);
		}
	}

	/**
	 * \ingroup Container
	 */
	public GObject find(Class objClass)
	{
		Iterator<GObject> iterator = iterator();
		while (iterator.hasNext())
		{
			GObject obj = iterator.next();
			if (obj.getClass().equals(objClass))
			{
				return obj;
			}
		}
		return null;
	}

	/**
	 * \ingroup Container
	 */
	public GObject extract(Class objClass)
	{
		GObject obj = find(objClass);
		// if (obj!=null)
		super.remove(obj);
		fireListeners();
		return obj;
	}

	public GObject in(int canvasX, int canvasY, GCoordSystem gcs)
	{
		Iterator<GObject> iterator = iterator();
		while (iterator.hasNext())
		{
			GObject obj = iterator.next();
			if (obj.in(canvasX, canvasY, gcs))
				return obj;
		}
		return null;
	}
}
